<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\View\Content;

use Comdev\Component\Onecore\Site\Helper\ContentHelper;
use Comdev\Component\Onecore\Site\Helper\CustomFieldsHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\MVC\View\GenericDataException;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * HTML Content View class for the OneCore component
 *
 * @since  1.0.0
 */
class HtmlView extends BaseHtmlView
{
	/**
	 * An array of items
	 *
	 * @var  array
	 */
	protected $items;

	/**
	 * The pagination object
	 *
	 * @var  \Joomla\CMS\Pagination\Pagination
	 */
	protected $pagination;

	/**
	 * The model state
	 *
	 * @var   \Joomla\Registry\Registry
	 */
	protected $state;

	/**
	 * The page parameters
	 *
	 * @var    \Joomla\Registry\Registry|null
	 */
	protected $params = null;

	/**
	 * Current layout (grid or list)
	 *
	 * @var    string
	 */
	public $currentLayout = 'grid';

	/**
	 * Execute and display a template script.
	 *
	 * @param   string  $tpl  The name of the template file to parse; automatically searches through the template paths.
	 *
	 * @return  void
	 */
	public function display($tpl = null)
	{
		/** @var \Comdev\Component\Onecore\Site\Model\ContentModel $model */
		$model = $this->getModel('Content');
		
		$this->items = $model->getItems();
		$this->pagination = $model->getPagination();
		$this->state = $model->getState();
		$this->params = $this->state->get('params');

		// Process items - add images and categories
		foreach ($this->items as &$item) {
			// Get first image
			$item->image = ContentHelper::getFirstImage($item->images ?? '');
			
			// Get all categories for this item
			$item->categories = ContentHelper::getCategories($item->id);
		}

		// Attach custom fields for list context ("items") in both positions
		$ids = array_map(static fn($it) => (int) ($it->id ?? 0), $this->items ?: []);
		$ids = array_values(array_filter($ids));

		if (!empty($ids)) {
			// Check if custom fields should be filtered by searchable setting
			// This can come from component params (menu item) or URL parameter from module
			$app = Factory::getApplication();
			$input = $app->getInput();
			$showCustomFields = (int) $input->get('show_custom_fields', $this->params->get('show_custom_fields', 1), 'int');
			// If show_custom_fields is enabled (1), show only searchable fields
			// If disabled (0), don't show any custom fields
			$onlySearchable = ($showCustomFields === 1) ? true : false;
			
			// If show_custom_fields is 0, don't load any fields at all
			if ($showCustomFields === 0) {
				$fieldsUnderTitle = [];
				$fieldsUnderIntro = [];
			} else {
				$fieldsUnderTitle = CustomFieldsHelper::getFieldsForContentIds($ids, 'items', 'under_title', $onlySearchable);
				$fieldsUnderIntro = CustomFieldsHelper::getFieldsForContentIds($ids, 'items', 'under_intro', $onlySearchable);
			}

			foreach ($this->items as &$item) {
				$cid = (int) ($item->id ?? 0);
				$item->customfields_under_title = $fieldsUnderTitle[$cid] ?? [];
				$item->customfields_under_intro = $fieldsUnderIntro[$cid] ?? [];
			}
		}

		// Check for errors.
		if (\count($errors = $model->getErrors())) {
			throw new GenericDataException(implode("\n", $errors), 500);
		}

		// Get all items with locations for map (regardless of pagination)
		$this->allMapItems = $this->getAllItemsWithLocations($model);

		// Determine layout from params or request (default to grid)
		$app = Factory::getApplication();
		$input = $app->getInput();
		$layoutParam = $input->get('layout', $this->params->get('layout', 'grid'), 'string');
		
		// Store layout in view for use in template
		$this->currentLayout = $layoutParam;
		
		// Always use 'default' layout which will load grid or list inside
		$this->setLayout('default');

		parent::display($tpl);
	}

	/**
	 * Get all items with locations for map display (regardless of pagination)
	 *
	 * @param   \Comdev\Component\Onecore\Site\Model\ContentModel  $model  The content model
	 *
	 * @return  array  Array of items with latitude and longitude
	 *
	 * @since   1.0.0
	 */
	protected function getAllItemsWithLocations($model): array
	{
		$db = Factory::getDbo();
		$query = $db->createQuery();

		// Get current state filters
		$state = $model->getState();
		$categoryId = $state->get('filter.category_id');
		$categories = $state->get('filter.categories', []);
		$search = $state->get('filter.search');
		$dateFrom = $state->get('filter.date_from');
		$dateTo = $state->get('filter.date_to');
		$showFeatured = $state->get('filter.show_featured');
		$featured = $state->get('filter.featured');
		$author = $state->get('filter.author');
		$published = $state->get('filter.published', 1);
		$customFieldsFilters = $state->get('filter.custom_fields', []);

		// Build query similar to ContentModel but without pagination
		$query->select([
			$db->quoteName('a.id'),
			$db->quoteName('a.title'),
			$db->quoteName('a.alias'),
			$db->quoteName('addr.latitude', 'latitude'),
			$db->quoteName('addr.longitude', 'longitude'),
		])
			->from($db->quoteName('#__one_content', 'a'))
			->leftJoin($db->quoteName('#__one_content_addresses', 'addr'), $db->quoteName('addr.content_id') . ' = ' . $db->quoteName('a.id'))
			->where($db->quoteName('a.published') . ' = :published')
			->where($db->quoteName('addr.latitude') . ' IS NOT NULL')
			->where($db->quoteName('addr.longitude') . ' IS NOT NULL')
			->where($db->quoteName('addr.latitude') . ' != ' . $db->quote(''))
			->where($db->quoteName('addr.longitude') . ' != ' . $db->quote(''))
			->bind(':published', $published, ParameterType::INTEGER);

		// Apply same filters as in ContentModel
		if (is_numeric($categoryId)) {
			$query->where($db->quoteName('a.id') . ' IN (
				SELECT ' . $db->quoteName('content_id') . ' 
				FROM ' . $db->quoteName('#__one_content_categories') . ' 
				WHERE ' . $db->quoteName('category_id') . ' = :category_id
			)')
				->bind(':category_id', $categoryId, ParameterType::INTEGER);
		}

		if (!empty($categories) && is_array($categories)) {
			$categories = array_filter(array_map('intval', $categories));
			if (!empty($categories)) {
				$query->where($db->quoteName('a.id') . ' IN (
					SELECT ' . $db->quoteName('content_id') . ' 
					FROM ' . $db->quoteName('#__one_content_categories') . ' 
					WHERE ' . $db->quoteName('category_id') . ' IN (' . implode(',', $categories) . ')
				)');
			}
		}

		if (!empty($search)) {
			$searchTerm = '%' . $db->escape($search, true) . '%';
			$query->where('(' . $db->quoteName('a.title') . ' LIKE :search1 
				OR ' . $db->quoteName('a.introtext') . ' LIKE :search2
				OR ' . $db->quoteName('a.fulltext') . ' LIKE :search3)')
				->bind(':search1', $searchTerm)
				->bind(':search2', $searchTerm)
				->bind(':search3', $searchTerm);
		}

		if (!empty($dateFrom)) {
			$query->where($db->quoteName('a.created') . ' >= :date_from')
				->bind(':date_from', $dateFrom);
		}

		if (!empty($dateTo)) {
			$query->where($db->quoteName('a.created') . ' <= :date_to')
				->bind(':date_to', $dateTo);
		}

		// Featured filter (only from URL parameter, not from menu params)
		// show_featured from menu params is for sorting only, not filtering
		if ($featured == 1) {
			$query->where($db->quoteName('a.featured') . ' = 1');
		}

		if ($author > 0) {
			$query->where($db->quoteName('a.created_by') . ' = :author')
				->bind(':author', $author, ParameterType::INTEGER);
		}

		// Filter by access level
		$user = Factory::getUser();
		if (!$user->authorise('core.admin')) {
			$groups = $user->getAuthorisedViewLevels();
			$query->whereIn($db->quoteName('a.access'), $groups);
		}

		// Apply custom fields filters (same logic as in ContentModel)
		if (!empty($customFieldsFilters) && is_array($customFieldsFilters)) {
			// Get field types for all filtered fields
			$fieldIds = array_keys($customFieldsFilters);
			$fieldIds = array_filter(array_map('intval', $fieldIds));
			$fieldTypes = [];
			if (!empty($fieldIds)) {
				$typeQuery = $db->createQuery()
					->select([$db->quoteName('id'), $db->quoteName('type')])
					->from($db->quoteName('#__one_customfields'))
					->whereIn($db->quoteName('id'), $fieldIds);
				$db->setQuery($typeQuery);
				try {
					$typeResults = $db->loadObjectList();
					foreach ($typeResults as $typeResult) {
						$fieldTypes[(int) $typeResult->id] = (string) $typeResult->type;
					}
				} catch (\Exception $e) {
					// Ignore errors
				}
			}

			foreach ($customFieldsFilters as $fieldId => $fieldValue) {
				$fieldId = (int) $fieldId;
				if ($fieldId <= 0) {
					continue;
				}

				$fieldType = $fieldTypes[$fieldId] ?? 'input';

				// Build subquery to check if content has this custom field value
				$subQuery = $db->createQuery()
					->select('1')
					->from($db->quoteName('#__one_customfield_values', 'cfv'))
					->innerJoin(
						$db->quoteName('#__one_customfields', 'cff'),
						$db->quoteName('cff.id') . ' = ' . $db->quoteName('cfv.field_id')
					)
					->where($db->quoteName('cfv.content_id') . ' = ' . $db->quoteName('a.id'))
					->where($db->quoteName('cfv.field_id') . ' = :cf_field_id_' . $fieldId)
					->where($db->quoteName('cff.published') . ' = 1')
					->where($db->quoteName('cff.searchable') . ' = 1');

				// Handle different value types and field types
				if (is_array($fieldValue)) {
					// Multiple values (for multi-select)
					$fieldValue = array_filter(array_map('trim', $fieldValue));
					if (!empty($fieldValue)) {
						if ($fieldType === 'select' || $fieldType === 'multiselect') {
							// For select/multiselect fields, check if value contains any of the selected values
							// For multiselect, values are stored as JSON, so we need to check JSON contains
							if ($fieldType === 'multiselect') {
								// For multiselect, check if JSON array contains any of the selected values
								$jsonConditions = [];
								foreach ($fieldValue as $idx => $val) {
									$paramName = ':cf_value_' . $fieldId . '_' . $idx;
									// Check if JSON contains the value (for multiselect stored as JSON array)
									$jsonConditions[] = $db->quoteName('cfv.value') . ' LIKE ' . $paramName;
									$searchValue = '%' . $db->escape($val, true) . '%';
									$query->bind($paramName, $searchValue, ParameterType::STRING);
								}
								if (!empty($jsonConditions)) {
									$subQuery->where('(' . implode(' OR ', $jsonConditions) . ')');
								}
							} else {
								// For select fields, use exact match
								$subQuery->whereIn($db->quoteName('cfv.value'), $fieldValue);
							}
						} else {
							// For text fields, use LIKE for each value
							$likeConditions = [];
							foreach ($fieldValue as $idx => $val) {
								$paramName = ':cf_value_' . $fieldId . '_' . $idx;
								$likeConditions[] = $db->quoteName('cfv.value') . ' LIKE ' . $paramName;
								$val = (string) trim($val);
								$searchValue = '%' . $val . '%';
								$query->bind($paramName, $searchValue, ParameterType::STRING);
							}
							if (!empty($likeConditions)) {
								$subQuery->where('(' . implode(' OR ', $likeConditions) . ')');
							}
						}
						$query->where('EXISTS (' . (string) $subQuery . ')');
						$fieldIdParam = ':cf_field_id_' . $fieldId;
						$query->bind($fieldIdParam, $fieldId, ParameterType::INTEGER);
					}
				} else {
					// Single value
					$fieldValue = trim($fieldValue);
					if (!empty($fieldValue)) {
						if ($fieldType === 'select') {
							// For select fields, use exact match
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' = ' . $valueParam);
							$query->bind($valueParam, $fieldValue);
						} elseif ($fieldType === 'multiselect') {
							// For multiselect, check if JSON contains the value
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' LIKE ' . $valueParam);
							$searchValue = '%' . $db->escape($fieldValue, true) . '%';
							$query->bind($valueParam, $searchValue, ParameterType::STRING);
						} else {
							// For text fields (input, textarea), use LIKE
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' LIKE ' . $valueParam);
							$fieldValue = (string) trim($fieldValue);
							$searchValue = '%' . $fieldValue . '%';
							$query->bind($valueParam, $searchValue, ParameterType::STRING);
						}
						$query->where('EXISTS (' . (string) $subQuery . ')');
						$fieldIdParam = ':cf_field_id_' . $fieldId;
						$query->bind($fieldIdParam, $fieldId, ParameterType::INTEGER);
					}
				}
			}
		}

		$db->setQuery($query);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}
}
