<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Service\Router;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Component\Router\Rules\RulesInterface;
use Joomla\CMS\Component\Router\RouterView;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

	/**
	 * Rule to find menu item for content/events list views and strip view from SEF URL.
	 *
	 * @since  1.0.0
	 */
class ContentMenuRule implements RulesInterface
{
	/**
	 * Router this rule belongs to
	 *
	 * @var   RouterView
	 */
	protected $router;

	/**
	 * Class constructor.
	 *
	 * @param   RouterView  $router  Router this rule belongs to
	 */
	public function __construct(RouterView $router)
	{
		$this->router = $router;
	}

	/**
	 * Finds the right Itemid when not set: content/events list views, item/event single views.
	 * Checks direct com_onecore items first, then aliases pointing to them.
	 *
	 * @param   array  &$query  The query array to process
	 *
	 * @return  void
	 */
	public function preprocess(&$query)
	{
		if (isset($query['Itemid']) || !isset($query['view'])) {
			return;
		}
		$menu = $this->router->app->getMenu();
		$need = $query['view'];

		if (\in_array($need, ['content', 'events'], true)) {
			$this->matchListMenu($query, $menu, $need);
			return;
		}
		if ($need === 'item' || $need === 'event') {
			$this->matchItemMenu($query, $menu);
		}
	}

	/**
	 * Set Itemid for content/events list views.
	 *
	 * @param   array   &$query  Query to update
	 * @param   object  $menu   Menu
	 * @param   string  $need   'content' or 'events'
	 *
	 * @return  void
	 */
	private function matchListMenu(&$query, $menu, $need)
	{
		$component = ComponentHelper::getComponent('com_onecore');
		$items     = $menu->getItems(['component_id'], [$component->id]);
		foreach ($items as $item) {
			$check = $this->resolveMenuItem($menu, $item);
			if (!$check || !$this->isOnecoreItem($check)) {
				continue;
			}
			$hasView = isset($check->query['view']);
			if ($hasView && $check->query['view'] === $need) {
				$query['Itemid'] = (int) $item->id;
				return;
			}
			if (!$hasView && $need === 'content') {
				$query['Itemid'] = (int) $item->id;
				return;
			}
		}
		$aliasItems = $menu->getItems(['type'], ['alias']);
		foreach ($aliasItems as $item) {
			$check = $this->resolveMenuItem($menu, $item);
			if (!$check || !$this->isOnecoreItem($check)) {
				continue;
			}
			$hasView = isset($check->query['view']);
			if ($hasView && $check->query['view'] === $need) {
				$query['Itemid'] = (int) $item->id;
				return;
			}
			if (!$hasView && $need === 'content') {
				$query['Itemid'] = (int) $item->id;
				return;
			}
		}
	}

	/**
	 * Set Itemid for item/event view (e.g. menu "single" with view=item/event or com_onecore without view).
	 *
	 * @param   array   &$query  Query to update
	 * @param   object  $menu   Menu
	 *
	 * @return  void
	 */
	private function matchItemMenu(&$query, $menu)
	{
		$component = ComponentHelper::getComponent('com_onecore');
		$items     = $menu->getItems(['component_id'], [$component->id]);
		foreach ($items as $item) {
			$check = $this->resolveMenuItem($menu, $item);
			if (!$check || !$this->isOnecoreItem($check)) {
				continue;
			}
			// Match: view=item/event OR view=content/events (item/event is child) OR no view (assume base)
			$hasView = isset($check->query['view']);
			if (($hasView && \in_array($check->query['view'], ['item', 'content', 'event', 'events'], true)) || (!$hasView)) {
				$query['Itemid'] = (int) $item->id;
				return;
			}
		}
		$aliasItems = $menu->getItems(['type'], ['alias']);
		foreach ($aliasItems as $item) {
			$check = $this->resolveMenuItem($menu, $item);
			if (!$check || !$this->isOnecoreItem($check)) {
				continue;
			}
			$hasView = isset($check->query['view']);
			if (($hasView && \in_array($check->query['view'], ['item', 'content', 'event', 'events'], true)) || (!$hasView)) {
				$query['Itemid'] = (int) $item->id;
				return;
			}
		}
	}

	/** Layout segments for content view (SEF path instead of ?layout=) */
	private const LAYOUT_SEGMENTS = ['grid', 'list'];

	/**
	 * Parse layout segment (e.g. /grid, /list) for content view.
	 *
	 * @param   array  &$segments  The URL segments to parse
	 * @param   array  &$vars      The vars that result from the segments
	 *
	 * @return  void
	 */
	public function parse(&$segments, &$vars)
	{
		if (\count($segments) === 0) {
			return;
		}
		$first = $segments[0];
		
		// Get views to check registered layouts
		$views = $this->router->getViews();
		
		// Check if segment is a registered layout for content or events view
		$isLayout = false;
		$targetView = null;
		
		if (isset($views['content']) && \in_array($first, $views['content']->layouts, true)) {
			$isLayout = true;
			$targetView = 'content';
		} elseif (isset($views['events']) && \in_array($first, $views['events']->layouts, true)) {
			$isLayout = true;
			$targetView = 'events';
		} elseif (\in_array($first, self::LAYOUT_SEGMENTS, true)) {
			// Fallback to hardcoded list for backward compatibility
			$isLayout = true;
			// Default to content view if not set
			$targetView = $vars['view'] ?? 'content';
		}
		
		// If segment is a layout segment (grid/list), parse it as layout for appropriate view
		if ($isLayout && $targetView) {
			// IMPORTANT: Set view and layout for layout segments
			// even if MenuRules set a different view (e.g., from default Home menu)
			// This ensures /component/onecore/list works correctly
			$vars['view'] = $targetView;
			$vars['layout'] = $first;
			array_shift($segments);
			// Return early to prevent StandardRules from trying to parse this segment
			return;
		}
	}

	/**
	 * Resolve menu item: if alias, return target; otherwise return item.
	 *
	 * @param   object  $menu  Menu (getMenu / getItems)
	 * @param   object  $item  Menu item
	 *
	 * @return  object|null  Resolved menu item or null
	 */
	private function resolveMenuItem($menu, $item)
	{
		if (!$item) {
			return null;
		}
		if ($item->type !== 'alias') {
			return $item;
		}
		$targetId = (int) $item->getParams()->get('aliasoptions');
		if (!$targetId) {
			return $item;
		}
		$target = $menu->getItem($targetId);

		return $target ?: $item;
	}

	/**
	 * Check if menu item is com_onecore (use resolved item).
	 *
	 * @param   object  $item  Menu item (or resolved target)
	 *
	 * @return  bool
	 */
	private function isOnecoreItem($item)
	{
		if (!$item) {
			return false;
		}
		$opt = $item->query['option'] ?? $item->component ?? '';

		return $opt === 'com_onecore';
	}

	/**
	 * Build method to prevent NomenuRules from adding view to URL
	 * This is called BEFORE StandardRules and NomenuRules
	 *
	 * @param   array  &$query     The vars that should be converted
	 * @param   array  &$segments  The URL segments to create
	 *
	 * @return  void
	 */
	public function build(&$query, &$segments)
	{
		// For item view, ALWAYS try to find Itemid if not set
		if (!isset($query['Itemid']) && isset($query['view']) && $query['view'] === 'item') {
			$menu = $this->router->app->getMenu();
			$this->matchItemMenu($query, $menu);
			
			// If still no Itemid found, try to find any com_onecore menu item as fallback
			if (!isset($query['Itemid'])) {
				$component = \Joomla\CMS\Component\ComponentHelper::getComponent('com_onecore');
				$items = $menu->getItems(['component_id'], [$component->id]);
				if (!empty($items) && isset($items[0]->id)) {
					// Use first available com_onecore menu item - ensure it's an integer
					$query['Itemid'] = (int) $items[0]->id;
				}
			}
		}
		
		// For event view, ALWAYS try to find Itemid if not set
		if (!isset($query['Itemid']) && isset($query['view']) && $query['view'] === 'event') {
			$menu = $this->router->app->getMenu();
			$component = \Joomla\CMS\Component\ComponentHelper::getComponent('com_onecore');
			$items = $menu->getItems(['component_id'], [$component->id]);
			if (!empty($items) && isset($items[0]->id)) {
				// Use first available com_onecore menu item - ensure it's an integer
				$query['Itemid'] = (int) $items[0]->id;
			}
		}
		
		if (!isset($query['Itemid'], $query['view'])) {
			return;
		}
		
		// Ensure Itemid is an integer (not array)
		if (is_array($query['Itemid'])) {
			return;
		}
		
		$menu  = $this->router->app->getMenu();
		$item  = $menu->getItem((int) $query['Itemid']);
		if (!$item) {
			return;
		}
		$check = $this->resolveMenuItem($menu, $item);
		if (!$check || !$this->isOnecoreItem($check)) {
			return;
		}

		if ($query['view'] === 'item') {
			$this->buildItem($query, $segments, $check);
			return;
		}

		if ($query['view'] === 'event') {
			$this->buildEvent($query, $segments, $check);
			return;
		}

		$listViews = ['content', 'events'];
		if (!\in_array($query['view'], $listViews, true)) {
			return;
		}
		$match = (isset($check->query['view']) && $check->query['view'] === $query['view'])
			|| !isset($check->query['view']);
		if (!$match) {
			return;
		}
		unset($query['view']);
		$layout = $query['layout'] ?? 'default';
		if (\in_array($layout, self::LAYOUT_SEGMENTS, true)) {
			$segments[] = $layout;
			unset($query['layout']);
		}
	}

	/**
	 * Build SEF segment for item view: /single/listing-alias
	 *
	 * @param   array   &$query    Query (view, id); mutated on success
	 * @param   array   &$segments Segments to append to
	 * @param   object  $check    Resolved menu item (view=item or com_onecore without view)
	 *
	 * @return  void
	 */
	private function buildItem(&$query, &$segments, $check)
	{
		// Menu item can have:
		// 1. view=item (direct item menu)
		// 2. view=content (item is child of content, use content menu as base)
		// 3. no view but com_onecore (assume item view base)
		$hasView = isset($check->query['view']);
		$menuView = $hasView ? $check->query['view'] : null;
		
		// If menu has view but it's not 'item' or 'content', skip
		if ($hasView && !\in_array($menuView, ['item', 'content'], true)) {
			return;
		}
		
		// If menu has no view but is not com_onecore, skip
		if (!$hasView && !$this->isOnecoreItem($check)) {
			return;
		}
		
		// Get item ID from query
		$id = $query['id'] ?? null;
		if ($id === null || $id === '') {
			return;
		}
		
		// Get segment from router
		if (!\is_callable([$this->router, 'getItemSegment'])) {
			return;
		}
		$path = $this->router->getItemSegment($id, $query);
		if ($path === [] || !\is_array($path) || \count($path) === 0) {
			return;
		}
		
		// Extract segment value (alias) - get the value from array
		// getItemSegment returns [$pk => $alias], so we need to get the value
		$seg = null;
		foreach ($path as $key => $value) {
			$seg = $value;
			break; // Get first value
		}
		
		if ($seg === null || $seg === '' || !\is_string($seg)) {
			return;
		}
		
		// Add segment and remove view/id from query to prevent StandardRules/NomenuRules from processing
		$segments[] = $seg;
		unset($query['view'], $query['id']);
	}

	/**
	 * Build SEF segment for event view: /events/event-alias
	 *
	 * @param   array   &$query    Query (view, id); mutated on success
	 * @param   array   &$segments Segments to append to
	 * @param   object  $check    Resolved menu item
	 *
	 * @return  void
	 */
	private function buildEvent(&$query, &$segments, $check)
	{
		// Get event ID from query
		$id = $query['id'] ?? null;
		if ($id === null || $id === '') {
			return;
		}

		// Get segment from router
		if (!\is_callable([$this->router, 'getEventSegment'])) {
			return;
		}

		$path = $this->router->getEventSegment($id, $query);
		if ($path === [] || !\is_array($path) || \count($path) === 0) {
			return;
		}

		// Extract segment value (alias) - get the value from array
		// getEventSegment returns [$pk => $alias], so we need to get the value
		$seg = null;
		foreach ($path as $key => $value) {
			$seg = $value;
			break; // Get first value
		}

		if ($seg === null || $seg === '' || !\is_string($seg)) {
			return;
		}

		// Add segment and remove view/id from query to prevent StandardRules/NomenuRules from processing
		$segments[] = $seg;
		unset($query['view'], $query['id']);
	}
}
