<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Service;

use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Multilanguage;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * OneCore Component Category Tree
 *
 * @since  1.0.0
 */
class Category extends Categories
{
	/**
	 * Class constructor
	 *
	 * @param   array  $options  Array of options
	 *
	 * @since   1.0.0
	 */
	/**
	 * Store original countItems setting
	 *
	 * @var    int
	 * @since  1.0.0
	 */
	private $_countItems = 0;

	/**
	 * Class constructor
	 *
	 * @param   array  $options  Array of options
	 *
	 * @since   1.0.0
	 */
	public function __construct($options = [])
	{
		$options['table']      = '#__one_content';
		$options['extension']  = 'com_onecore';
		$options['statefield'] = 'published';
		// Disable default countItems because we use many-to-many relationship
		// We'll count items manually after loading
		$countItems = $options['countItems'] ?? 0;
		$options['countItems'] = 0;

		parent::__construct($options);

		// Store original countItems setting
		$this->_countItems = $countItems;
	}

	/**
	 * Load all categories flag
	 *
	 * @var    bool
	 * @since  1.0.0
	 */
	private $_allCategoriesLoaded = false;

	/**
	 * Load method - override to use #__categories table with extension = com_onecore
	 *
	 * @param   int|string  $id  Id of category to load
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function _load($id)
	{
		// Prevent infinite loop - if already checked this id, return
		if (isset($this->_checkedCategories[$id])) {
			return;
		}

		try {
			$db = $this->getDatabase();
		} catch (\Joomla\Database\Exception\DatabaseNotFoundException $e) {
			$db = Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class);
		}

		$app       = Factory::getApplication();
		$user      = Factory::getUser();
		$extension = $this->_extension;

		if ($id !== 'root') {
			$id = (int) $id;

			if ($id === 0) {
				$id = 'root';
			}
		}

		// Record that has this $id has been checked
		$this->_checkedCategories[$id] = true;

		// Load ALL categories at once to avoid recursive calls
		if (!$this->_allCategoriesLoaded) {
			$this->_loadAllCategories($db, $user, $extension);
			$this->_allCategoriesLoaded = true;
		}

		// Now return the requested node
		if ($id === 'root') {
			if (!isset($this->_nodes['root'])) {
				$rootNode = new \Joomla\CMS\Categories\CategoryNode((object)['id' => 'root', 'title' => 'ROOT'], $this);
				// Use reflection to set _children array directly
				$reflection = new \ReflectionClass($rootNode);
				$childrenProperty = $reflection->getProperty('_children');
				$childrenProperty->setAccessible(true);
				$rootChildren = [];
				foreach ($this->_nodes as $node) {
					if (isset($node->parent_id) && $node->parent_id == 0) {
						$rootChildren[] = $node;
						$node->setParent($rootNode);
					}
				}
				$childrenProperty->setValue($rootNode, $rootChildren);
				$rootNode->setAllLoaded();
				$this->_nodes['root'] = $rootNode;
			}
		} elseif (!isset($this->_nodes[$id])) {
			$this->_nodes[$id] = null;
		}
	}

	/**
	 * Load all categories from database
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db         Database object
	 * @param   \Joomla\CMS\User\User                $user       User object
	 * @param   string                                $extension  Extension name
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function _loadAllCategories($db, $user, $extension)
	{
		// Build query for #__categories table with extension = com_onecore
		$query = $db->createQuery()
			->select(
				[
					$db->quoteName('c.id'),
					$db->quoteName('c.access'),
					$db->quoteName('c.alias'),
					$db->quoteName('c.checked_out'),
					$db->quoteName('c.checked_out_time'),
					$db->quoteName('c.created_time'),
					$db->quoteName('c.created_user_id'),
					$db->quoteName('c.description'),
					$db->quoteName('c.extension'),
					$db->quoteName('c.hits'),
					$db->quoteName('c.language'),
					$db->quoteName('c.parent_id'),
					$db->quoteName('c.published'),
					$db->quoteName('c.title'),
					$db->quoteName('c.modified_user_id'),
					$db->quoteName('c.params'),
					$db->quoteName('c.metakey'),
					$db->quoteName('c.metadesc'),
					$db->quoteName('c.metadata'),
					$db->quoteName('c.note'),
					$db->quoteName('c.version'),
					$db->quoteName('c.asset_id'),
					$db->quoteName('c.level'),
					$db->quoteName('c.lft'),
					$db->quoteName('c.rgt'),
					$db->quoteName('c.path'),
				]
			)
			->from($db->quoteName('#__categories', 'c'))
			->where($db->quoteName('c.extension') . ' = :extension')
			->bind(':extension', $extension);

		if ($this->_options['access']) {
			$groups = $user->getAuthorisedViewLevels();
			$query->whereIn($db->quoteName('c.access'), $groups);
		}

		if ($this->_options['published'] == 1) {
			$query->where($db->quoteName('c.published') . ' = 1');
		}

		// Order by lft (nested set structure) - this is the standard way to order categories in Joomla
		$query->order($db->quoteName('c.lft') . ' ASC');

		// Count items if requested
		if ($this->_countItems == 1) {
			$subQuery = $db->createQuery()
				->select('COUNT(DISTINCT ' . $db->quoteName('i.id') . ')')
				->from($db->quoteName('#__one_content', 'i'))
				->innerJoin(
					$db->quoteName('#__one_content_categories', 'cc'),
					$db->quoteName('cc.content_id') . ' = ' . $db->quoteName('i.id')
				)
				->where($db->quoteName('cc.category_id') . ' = ' . $db->quoteName('c.id'));

			if ($this->_options['published'] == 1) {
				$subQuery->where($db->quoteName('i.published') . ' = 1');
			}

			if ($this->_options['currentlang'] !== 0) {
				$subQuery->where(
					$db->quoteName('i.language')
					. ' IN (' . implode(',', $subQuery->bindArray([$this->_options['currentlang'], '*'], ParameterType::STRING)) . ')'
				);
			}

			$query->select('(' . $subQuery . ') AS ' . $db->quoteName('numitems'));
		} else {
			$query->select('0 AS ' . $db->quoteName('numitems'));
		}

		// Get all results
		$db->setQuery($query);
		$results = $db->loadObjectList('id');

		if (\count($results)) {
			// Create CategoryNode objects for all categories
			foreach ($results as $result) {
				if (!isset($this->_nodes[$result->id])) {
					$node = new \Joomla\CMS\Categories\CategoryNode($result, $this);
					// Use reflection to set _children array directly to prevent recursive loading
					$reflection = new \ReflectionClass($node);
					$childrenProperty = $reflection->getProperty('_children');
					$childrenProperty->setAccessible(true);
					$childrenProperty->setValue($node, []);
					$this->_nodes[$result->id] = $node;
				}
			}

			// Build parent-child relationships
			foreach ($this->_nodes as $node) {
				if (isset($node->parent_id) && $node->parent_id > 0 && isset($this->_nodes[$node->parent_id])) {
					// Use reflection to add child directly to _children array
					$reflection = new \ReflectionClass($this->_nodes[$node->parent_id]);
					$childrenProperty = $reflection->getProperty('_children');
					$childrenProperty->setAccessible(true);
					$children = $childrenProperty->getValue($this->_nodes[$node->parent_id]);
					$children[] = $node;
					$childrenProperty->setValue($this->_nodes[$node->parent_id], $children);
					// Set parent reference
					$node->setParent($this->_nodes[$node->parent_id]);
				}
			}

			// Mark all nodes as fully loaded AFTER setting children to prevent recursive calls
			foreach ($this->_nodes as $node) {
				$node->setAllLoaded();
			}
		}
	}
}
