<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Model;

use Comdev\Component\Onecore\Site\Helper\CustomFieldsHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Events Component List Model
 *
 * @since  1.0.0
 */
class EventsModel extends ListModel
{
	/**
	 * Model context string.
	 *
	 * @var  string
	 */
	protected $_context = 'com_onecore.events';

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'title', 'a.title',
				'alias', 'a.alias',
				'state', 'a.published',
				'event_start', 'a.event_start',
				'event_end', 'a.event_end',
				'created', 'a.created',
				'created_by', 'a.created_by',
				'ordering', 'a.ordering',
				'language', 'a.language',
				'hits', 'a.hits',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 */
	protected function populateState($ordering = 'a.event_start', $direction = 'ASC')
	{
		$app = Factory::getApplication();

		// Load the parameters.
		$params = $app->getParams();
		$this->setState('params', $params);

		// Get category ID from menu item (request.id) or input
		$active = $app->getMenu()->getActive();
		$categoryId = 0;
		
		// First, try to get category_id from active menu item's request.id
		if ($active && isset($active->query['view']) && $active->query['view'] === 'events') {
			if (isset($active->query['id']) && $active->query['id'] > 0) {
				$categoryId = (int) $active->query['id'];
			}
		}
		
		// If no category_id from menu item, check URL parameter
		if ($categoryId === 0) {
			$urlId = $app->getInput()->getInt('id', 0);
			if ($urlId > 0) {
				$categoryId = $urlId;
			}
		}
		
		if ($categoryId > 0) {
			$this->setState('filter.category_id', $categoryId);
		}

		// List state information
		$input = $app->getInput();
		$defaultLimit = $app->getUserStateFromRequest('global.list.limit', 'limit', $app->get('list_limit', 20), 'uint');
		$limit = $input->getInt('limit', $defaultLimit);
		$this->setState('list.limit', $limit);

		$limitstart = $input->get('limitstart', 0, 'uint');
		$this->setState('list.start', $limitstart);

		// Filter by published state
		$this->setState('filter.published', 1);

		// Search filter
		$search = $input->getString('search', '');
		if (!empty($search)) {
			$this->setState('filter.search', $search);
		}

		// Category filter (multi-select)
		$categories = $input->get('category', [], 'array');
		if (!empty($categories) && is_array($categories)) {
			$categories = array_filter(array_map('intval', $categories));
			if (!empty($categories)) {
				$this->setState('filter.categories', $categories);
			}
		}

		// Event date filters (event_date_from, event_date_to)
		$dateFrom = $input->getString('event_date_from', $input->getString('date_from', ''));
		$dateTo = $input->getString('event_date_to', $input->getString('date_to', ''));
		if (!empty($dateFrom)) {
			$this->setState('filter.date_from', $dateFrom);
		}
		if (!empty($dateTo)) {
			$this->setState('filter.date_to', $dateTo);
		}

		// Featured filter
		$featured = $input->getInt('featured', 0);
		if ($featured) {
			$this->setState('filter.featured', 1);
		}

		// Custom fields filters (cf_* parameters)
		$customFieldsFilters = [];
		$allInput = $input->getArray();
		foreach ($allInput as $key => $value) {
			if (strpos($key, 'cf_') === 0) {
				$fieldId = (int) substr($key, 3);
				if ($fieldId > 0 && !empty($value)) {
					$customFieldsFilters[$fieldId] = is_array($value) ? $value : $value;
				}
			}
		}
		if (!empty($customFieldsFilters)) {
			$this->setState('filter.custom_fields', $customFieldsFilters);
		}

		// Ordering - handle sort parameter (newest, oldest, a-z, z-a, date)
		$sortParam = $input->getString('sort', '');
		if (!empty($sortParam)) {
			switch ($sortParam) {
				case 'newest':
					$orderCol = 'a.event_start';
					$orderDirn = 'DESC';
					break;
				case 'oldest':
					$orderCol = 'a.event_start';
					$orderDirn = 'ASC';
					break;
				case 'a-z':
					$orderCol = 'a.title';
					$orderDirn = 'ASC';
					break;
				case 'z-a':
					$orderCol = 'a.title';
					$orderDirn = 'DESC';
					break;
				case 'date':
					$orderCol = 'a.event_start';
					$orderDirn = 'ASC';
					break;
				default:
					$orderCol = $ordering;
					$orderDirn = $direction;
			}
		} else {
			// Default to event_start ASC for events
			$orderCol = 'a.event_start';
			$orderDirn = 'ASC';
		}

		$this->setState('list.ordering', $orderCol);
		$this->setState('list.direction', $orderDirn);
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('list.start');
		$id .= ':' . $this->getState('list.limit');
		$id .= ':' . $this->getState('list.ordering');
		$id .= ':' . $this->getState('list.direction');
		$id .= ':' . $this->getState('filter.published');
		$id .= ':' . $this->getState('filter.category_id');
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . serialize($this->getState('filter.categories', []));
		$id .= ':' . $this->getState('filter.date_from');
		$id .= ':' . $this->getState('filter.date_to');
		$id .= ':' . $this->getState('filter.featured');
		$id .= ':' . serialize($this->getState('filter.custom_fields', []));
		
		// Add sort parameter to cache ID if exists
		$app = Factory::getApplication();
		$sortParam = $app->getInput()->getString('sort', '');
		if (!empty($sortParam)) {
			$id .= ':' . $sortParam;
		}

		return parent::getStoreId($id);
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\QueryInterface
	 */
	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->createQuery();

		// Ensure custom fields schema is present
		CustomFieldsHelper::ensureReady();

		// Select the required fields from the events table
		$query->select(
			$db->quoteName(
				[
					'a.id',
					'a.title',
					'a.alias',
					'a.description',
					'a.published',
					'a.featured',
					'a.event_start',
					'a.event_end',
					'a.created',
					'a.created_by',
					'a.modified',
					'a.hits',
					'a.language',
					'a.images',
				]
			)
		)
			->from($db->quoteName('#__one_events', 'a'));

		// Join on user table
		$query->select($db->quoteName('u.name', 'author'))
			->leftJoin($db->quoteName('#__users', 'u'), $db->quoteName('u.id') . ' = ' . $db->quoteName('a.created_by'));

		// Join on address table
		$query->select($db->quoteName('addr.latitude', 'latitude'))
			->select($db->quoteName('addr.longitude', 'longitude'))
			->select($db->quoteName('addr.address', 'address'))
			->leftJoin($db->quoteName('#__one_event_addresses', 'addr'), $db->quoteName('addr.event_id') . ' = ' . $db->quoteName('a.id'));

		// Filter by category (single category from menu/URL)
		$categoryId = $this->getState('filter.category_id');
		if (is_numeric($categoryId)) {
			$query->where($db->quoteName('a.id') . ' IN (
				SELECT ' . $db->quoteName('event_id') . ' 
				FROM ' . $db->quoteName('#__one_event_categories') . ' 
				WHERE ' . $db->quoteName('category_id') . ' = :category_id
			)')
				->bind(':category_id', $categoryId, ParameterType::INTEGER);
		}

		// Filter by categories (multi-select from search form)
		$categories = $this->getState('filter.categories');
		if (!empty($categories) && is_array($categories)) {
			$categories = array_filter(array_map('intval', $categories));
			if (!empty($categories)) {
				$query->where($db->quoteName('a.id') . ' IN (
					SELECT ' . $db->quoteName('event_id') . ' 
					FROM ' . $db->quoteName('#__one_event_categories') . ' 
					WHERE ' . $db->quoteName('category_id') . ' IN (' . implode(',', $categories) . ')
				)');
			}
		}

		// Search filter (text search in title, description)
		$search = $this->getState('filter.search');
		if (!empty($search)) {
			$searchTerm = '%' . $db->escape($search, true) . '%';
			$query->where('(' . $db->quoteName('a.title') . ' LIKE :search1 
				OR ' . $db->quoteName('a.description') . ' LIKE :search2
				OR EXISTS (
					SELECT 1
					FROM ' . $db->quoteName('#__one_customfield_values', 'cfv') . '
					INNER JOIN ' . $db->quoteName('#__one_customfields', 'cff') . ' ON ' . $db->quoteName('cff.id') . ' = ' . $db->quoteName('cfv.field_id') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'fcat') . ' ON ' . $db->quoteName('fcat.id') . ' = ' . $db->quoteName('cff.category_id') . '
					INNER JOIN ' . $db->quoteName('#__one_event_categories', 'eec') . ' ON ' . $db->quoteName('eec.event_id') . ' = ' . $db->quoteName('a.id') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'ecat') . ' ON ' . $db->quoteName('ecat.id') . ' = ' . $db->quoteName('eec.category_id') . '
					WHERE ' . $db->quoteName('cfv.event_id') . ' = ' . $db->quoteName('a.id') . '
					  AND ' . $db->quoteName('cff.published') . ' = 1
					  AND ' . $db->quoteName('cff.searchable') . ' = 1
					  AND ' . $db->quoteName('fcat.lft') . ' <= ' . $db->quoteName('ecat.lft') . '
					  AND ' . $db->quoteName('fcat.rgt') . ' >= ' . $db->quoteName('ecat.rgt') . '
					  AND ' . $db->quoteName('cfv.value') . ' LIKE :search3
				)
			)')
				->bind(':search1', $searchTerm)
				->bind(':search2', $searchTerm)
				->bind(':search3', $searchTerm);
		}

		// Event date filters (filter by event_start and event_end)
		$dateFrom = $this->getState('filter.date_from');
		$dateTo = $this->getState('filter.date_to');
		if (!empty($dateFrom)) {
			$query->where($db->quoteName('a.event_start') . ' >= :date_from')
				->bind(':date_from', $dateFrom);
		}
		if (!empty($dateTo)) {
			$query->where($db->quoteName('a.event_end') . ' <= :date_to')
				->bind(':date_to', $dateTo);
		}

		// Featured filter
		$featured = $this->getState('filter.featured');
		if ($featured) {
			$query->where($db->quoteName('a.featured') . ' = 1');
		}

		// Custom fields filters
		$customFieldsFilters = $this->getState('filter.custom_fields', []);
		if (!empty($customFieldsFilters) && is_array($customFieldsFilters)) {
			// Get field types for all filtered fields
			$fieldIds = array_keys($customFieldsFilters);
			$fieldIds = array_filter(array_map('intval', $fieldIds));
			$fieldTypes = [];
			if (!empty($fieldIds)) {
				$typeQuery = $db->createQuery()
					->select([$db->quoteName('id'), $db->quoteName('type')])
					->from($db->quoteName('#__one_customfields'))
					->whereIn($db->quoteName('id'), $fieldIds);
				$db->setQuery($typeQuery);
				try {
					$typeResults = $db->loadObjectList();
					foreach ($typeResults as $typeResult) {
						$fieldTypes[(int) $typeResult->id] = (string) $typeResult->type;
					}
				} catch (\Exception $e) {
					// Ignore errors
				}
			}

			foreach ($customFieldsFilters as $fieldId => $fieldValue) {
				$fieldId = (int) $fieldId;
				if ($fieldId <= 0) {
					continue;
				}

				$fieldType = $fieldTypes[$fieldId] ?? 'input';

				// Build subquery to check if event has this custom field value
				$subQuery = $db->createQuery()
					->select('1')
					->from($db->quoteName('#__one_customfield_values', 'cfv'))
					->innerJoin(
						$db->quoteName('#__one_customfields', 'cff'),
						$db->quoteName('cff.id') . ' = ' . $db->quoteName('cfv.field_id')
					)
					->where($db->quoteName('cfv.event_id') . ' = ' . $db->quoteName('a.id'))
					->where($db->quoteName('cfv.field_id') . ' = :cf_field_id_' . $fieldId)
					->where($db->quoteName('cff.published') . ' = 1')
					->where($db->quoteName('cff.searchable') . ' = 1');

				// Handle different value types and field types
				if (is_array($fieldValue)) {
					// Multiple values (for multi-select)
					$fieldValue = array_filter(array_map('trim', $fieldValue));
					if (!empty($fieldValue)) {
						if ($fieldType === 'select' || $fieldType === 'multiselect') {
							if ($fieldType === 'multiselect') {
								$jsonConditions = [];
								foreach ($fieldValue as $idx => $val) {
									$paramName = ':cf_value_' . $fieldId . '_' . $idx;
									$jsonConditions[] = $db->quoteName('cfv.value') . ' LIKE ' . $paramName;
									$searchValue = '%' . $db->escape($val, true) . '%';
									$query->bind($paramName, $searchValue, ParameterType::STRING);
								}
								if (!empty($jsonConditions)) {
									$subQuery->where('(' . implode(' OR ', $jsonConditions) . ')');
								}
							} else {
								$subQuery->whereIn($db->quoteName('cfv.value'), $fieldValue);
							}
						} else {
							$likeConditions = [];
							foreach ($fieldValue as $idx => $val) {
								$paramName = ':cf_value_' . $fieldId . '_' . $idx;
								$likeConditions[] = $db->quoteName('cfv.value') . ' LIKE ' . $paramName;
								$val = (string) trim($val);
								$searchValue = '%' . $val . '%';
								$query->bind($paramName, $searchValue, ParameterType::STRING);
							}
							if (!empty($likeConditions)) {
								$subQuery->where('(' . implode(' OR ', $likeConditions) . ')');
							}
						}
						$query->where('EXISTS (' . (string) $subQuery . ')');
						$fieldIdParam = ':cf_field_id_' . $fieldId;
						$query->bind($fieldIdParam, $fieldId, ParameterType::INTEGER);
					}
				} else {
					// Single value
					$fieldValue = trim($fieldValue);
					if (!empty($fieldValue)) {
						if ($fieldType === 'select') {
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' = ' . $valueParam);
							$query->bind($valueParam, $fieldValue);
						} elseif ($fieldType === 'multiselect') {
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' LIKE ' . $valueParam);
							$searchValue = '%' . $db->escape($fieldValue, true) . '%';
							$query->bind($valueParam, $searchValue, ParameterType::STRING);
						} else {
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' LIKE ' . $valueParam);
							$fieldValue = (string) trim($fieldValue);
							$searchValue = '%' . $fieldValue . '%';
							$query->bind($valueParam, $searchValue, ParameterType::STRING);
						}
						$query->where('EXISTS (' . (string) $subQuery . ')');
						$fieldIdParam = ':cf_field_id_' . $fieldId;
						$query->bind($fieldIdParam, $fieldId, ParameterType::INTEGER);
					}
				}
			}
		}

		// Filter by published state
		$published = $this->getState('filter.published');
		if (is_numeric($published)) {
			$query->where($db->quoteName('a.published') . ' = :published')
				->bind(':published', $published, ParameterType::INTEGER);
		}

		// Filter by access level
		$user = Factory::getUser();
		if (!$user->authorise('core.admin')) {
			$groups = $user->getAuthorisedViewLevels();
			$query->whereIn($db->quoteName('a.access'), $groups);
		}

		// Add the list ordering clause
		$orderCol  = $this->state->get('list.ordering', 'a.event_start');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		// Ensure ordering column has table alias
		if ($orderCol === 'ordering' || $orderCol === 'event_start' || $orderCol === 'event_end') {
			$orderCol = 'a.' . $orderCol;
		}
		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}
}
