<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Model;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ItemModel as BaseItemModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Event Component Item Model
 *
 * @since  1.0.0
 */
class EventModel extends BaseItemModel
{
	/**
	 * Model context string.
	 *
	 * @var        string
	 */
	protected $_context = 'com_onecore.event';

	/**
	 * Method to auto-populate the model state.
	 *
	 * @return void
	 */
	protected function populateState()
	{
		$app = Factory::getApplication();

		// Load state from the request.
		$pk = $app->getInput()->getInt('id');
		$this->setState('event.id', $pk);

		$offset = $app->getInput()->getUint('limitstart');
		$this->setState('list.offset', $offset);

		// Load the parameters.
		$params = $app->getParams();
		$this->setState('params', $params);
	}

	/**
	 * Method to get event data.
	 *
	 * @param   integer  $pk  The id of the event.
	 *
	 * @return  object|boolean  Event data object on success, boolean false
	 */
	public function getItem($pk = null)
	{
		$pk = (int) ($pk ?: $this->getState('event.id'));

		if ($this->_item === null) {
			$this->_item = [];
		}

		if (!isset($this->_item[$pk])) {
			try {
				$db    = $this->getDatabase();
				$query = $db->createQuery();

				$query->select(
					$db->quoteName(
						[
							'a.id',
							'a.title',
							'a.alias',
							'a.description',
							'a.event_start',
							'a.event_end',
							'a.recur_event',
							'a.published',
							'a.featured',
							'a.access',
							'a.created',
							'a.created_by',
							'a.modified',
							'a.modified_by',
							'a.language',
							'a.images',
							'a.metakey',
							'a.metadesc',
							'a.metadata',
						]
					)
				)
					->from($db->quoteName('#__one_events', 'a'))
					->where($db->quoteName('a.id') . ' = :pk')
					->bind(':pk', $pk, ParameterType::INTEGER);

				// Join on user table.
				$query->select($db->quoteName('u.name', 'author'))
					->leftJoin($db->quoteName('#__users', 'u'), $db->quoteName('u.id') . ' = ' . $db->quoteName('a.created_by'));

				// Join on address table.
				$query->select($db->quoteName('addr.latitude', 'latitude'))
					->select($db->quoteName('addr.longitude', 'longitude'))
					->select($db->quoteName('addr.address', 'address'))
					->select($db->quoteName('addr.address_street', 'address_street'))
					->select($db->quoteName('addr.address_street_number', 'address_street_number'))
					->select($db->quoteName('addr.address_postal_code', 'address_postal_code'))
					->select($db->quoteName('addr.address_city', 'address_city'))
					->select($db->quoteName('addr.address_country', 'address_country'))
					->leftJoin($db->quoteName('#__one_event_addresses', 'addr'), $db->quoteName('addr.event_id') . ' = ' . $db->quoteName('a.id'));

				// Join on recurrences table for recurring event details
				$query->select($db->quoteName('rec.frequency', 'recur_frequency'))
					->select($db->quoteName('rec.interval', 'recur_interval'))
					->select($db->quoteName('rec.byday', 'recur_byday'))
					->select($db->quoteName('rec.until_date', 'recur_until'))
					->select($db->quoteName('rec.count', 'recur_count'))
					->leftJoin($db->quoteName('#__one_event_recurrences', 'rec'), $db->quoteName('rec.event_id') . ' = ' . $db->quoteName('a.id'));

				// Filter by access level.
				$user = Factory::getUser();
				if (!$user->authorise('core.admin')) {
					$groups = $user->getAuthorisedViewLevels();
					$query->whereIn($db->quoteName('a.access'), $groups);
				}

				// Filter by published state
				$query->where($db->quoteName('a.published') . ' = 1');

				$db->setQuery($query);
				$data = $db->loadObject();

				if (empty($data)) {
					throw new \Exception(\Joomla\CMS\Language\Text::_('COM_ONECORE_ERROR_ITEM_NOT_FOUND'), 404);
				}

				// Get categories for this event
				$categoriesQuery = $db->createQuery()
					->select([
						$db->quoteName('c.id'),
						$db->quoteName('c.title'),
						$db->quoteName('c.alias'),
					])
					->from($db->quoteName('#__one_event_categories', 'ec'))
					->innerJoin($db->quoteName('#__one_categories', 'c'), $db->quoteName('c.id') . ' = ' . $db->quoteName('ec.category_id'))
					->where($db->quoteName('ec.event_id') . ' = :event_id')
					->bind(':event_id', $pk, ParameterType::INTEGER);

				$db->setQuery($categoriesQuery);
				$data->categories = $db->loadObjectList() ?: [];

				// Get first image
				$data->image = \Comdev\Component\Onecore\Site\Helper\ContentHelper::getFirstImage($data->images ?? '');
				
				// Get all images
				$data->all_images = \Comdev\Component\Onecore\Site\Helper\ContentHelper::getAllImages($data->images ?? '');

				$this->_item[$pk] = $data;
			} catch (\Exception $e) {
				if ($e->getCode() == 404) {
					// Need to go through the error handler to allow Redirect to work.
					throw $e;
				} else {
					$this->setError($e);
					$this->_item[$pk] = false;
				}
			}
		}

		return $this->_item[$pk];
	}
}
