<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

\defined('_JEXEC') or die;


use Joomla\CMS\Factory;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Version;

class Com_OnecoreInstallerScript
{
	/**
	 * Simple logger – writes to onecore-install.log
	 *
	 * @param   string  $message  Log message
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function log(string $message): void
	{
		$file = JPATH_ADMINISTRATOR . '/logs/onecore-install.log';
		@file_put_contents(
			$file,
			'[' . date('c') . '] ' . $message . PHP_EOL,
			FILE_APPEND
		);
	}

	/**
	 * Function called before extension installation/update/removal procedure commences
	 *
	 * @param   string            $type    The type of change (install or discover_install, update, uninstall)
	 * @param   InstallerAdapter  $parent  The adapter calling this method
	 *
	 * @return  boolean  True on success
	 *
	 * @since   1.0.0
	 */
	public function preflight(string $type, InstallerAdapter $parent): bool
	{
		$this->log('PRELIGHT: ' . $type);

		// Minimalne sprawdzenie wersji Joomla
		$jVersion = new Version;
		if (version_compare($jVersion->getShortVersion(), '5.0.0', '<')) {
			$this->log('ERROR: Joomla < 5.0');
			return false;
		}

		// For uninstall, ensure instance_id is available before component is removed
		if ($type === 'uninstall') {
			$this->getOrCreateInstanceId(); // Ensure instance_id exists before uninstall
			$this->log('PREFLIGHT: Instance ID ensured for uninstall');
		}

		return true;
	}

	/**
	 * Function called after the extension is installed
	 *
	 * @param   InstallerAdapter  $parent  The adapter calling this method
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function install(InstallerAdapter $parent): void
	{
		$this->log('INSTALL called');
		
		try {
			$manifest = $parent->getManifest();
			$version = (string) $manifest->version;
			
			$instanceId = $this->getOrCreateInstanceId();
			$this->log('INSTALL: Instance ID=' . substr($instanceId, 0, 8) . '..., Version=' . $version);
			
			$this->sendInstallationStats('install', $instanceId, $version);
		} catch (\Exception $e) {
			$this->log('INSTALL ERROR: ' . $e->getMessage());
		}
	}

	/**
	 * Function called after the extension is updated
	 *
	 * @param   InstallerAdapter  $parent  The adapter calling this method
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function update(InstallerAdapter $parent): void
	{
		$this->log('UPDATE called');
		
		try {
			$manifest = $parent->getManifest();
			$version = (string) $manifest->version;
			
			$instanceId = $this->getOrCreateInstanceId();
			$this->log('UPDATE: Instance ID=' . substr($instanceId, 0, 8) . '..., Version=' . $version);
			
			$this->sendInstallationStats('install', $instanceId, $version);
		} catch (\Exception $e) {
			$this->log('UPDATE ERROR: ' . $e->getMessage());
		}
	}

	/**
	 * Function called after the extension is uninstalled
	 *
	 * @param   InstallerAdapter  $parent  The adapter calling this method
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function uninstall(InstallerAdapter $parent): void
	{
		$this->log('UNINSTALL called');
		
		try {
			$instanceId = $this->getInstanceId();
			
			if ($instanceId) {
				$version = $this->getComponentVersion();
				$this->log('UNINSTALL: Instance ID=' . substr($instanceId, 0, 8) . '..., Version=' . $version);
				
				$this->sendInstallationStats('uninstall', $instanceId, $version);
			} else {
				$this->log('UNINSTALL: No instance_id found');
			}
		} catch (\Exception $e) {
			$this->log('UNINSTALL ERROR: ' . $e->getMessage());
		}
	}

	/**
	 * Function called after extension installation/update/removal procedure commences
	 *
	 * @param   string            $type    The type of change (install or discover_install, update, uninstall)
	 * @param   InstallerAdapter  $parent  The adapter calling this method
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function postflight(string $type, InstallerAdapter $parent): void
	{
		$this->log('POSTFLIGHT: ' . $type);
		
		if ($type === 'install' || $type === 'update') {
			try {
				$db = Factory::getDbo();
				
				// Get component version from manifest
				$manifest = $parent->getManifest();
				$version = (string) $manifest->version;
				
				// Get existing params to preserve instance_id and user overrides
				$query = $db->getQuery(true)
					->select($db->quoteName('params'))
					->from($db->quoteName('#__extensions'))
					->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
					->where($db->quoteName('type') . ' = ' . $db->quote('component'));
				$db->setQuery($query);
				$paramsJson = $db->loadResult();
				
				// Parse existing params
				$params = $paramsJson ? json_decode($paramsJson, true) : [];
				if (!is_array($params)) {
					$params = [];
				}
				
				// Apply defaults for any missing keys (no need to "Save" Options after install)
				$defaults = $this->getDefaultComponentParams();
				foreach ($defaults as $key => $value) {
					if (!array_key_exists($key, $params)) {
						$params[$key] = $value;
					}
				}
				
				// Force component_version and license_type
				$params['component_version'] = $version;
				if (!isset($params['license_type'])) {
					$params['license_type'] = 'free';
				}
				
				// Update params
				$query = $db->getQuery(true)
					->update($db->quoteName('#__extensions'))
					->set($db->quoteName('params') . ' = ' . $db->quote(json_encode($params)))
					->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
					->where($db->quoteName('type') . ' = ' . $db->quote('component'));
				
				$db->setQuery($query)->execute();
				$this->log('POSTFLIGHT: Component params updated (defaults applied, instance_id preserved)');
			} catch (\Exception $e) {
				$this->log('POSTFLIGHT ERROR: ' . $e->getMessage());
			}
		}
	}

	/**
	 * Default component options (config.xml). Used on install/update so Options need not be saved.
	 *
	 * @return  array<string, mixed>
	 *
	 * @since   1.0.0
	 */
	private function getDefaultComponentParams(): array
	{
		return [
			'layout'                    => 'default',
			'show_title'                => 1,
			'linked_titles'             => 1,
			'show_intro_text'           => 1,
			'show_article_info_title'   => 1,
			'show_category'             => 1,
			'link_category'             => 1,
			'show_parent_category'      => 0,
			'show_create_date'          => 0,
			'show_item_sidebar'         => 1,
			'item_sidebar_position'     => 'onecore-item-sidebar',
			'item_sidebar_width'        => 4,
			'item_block_contact_show'   => 1,
			'item_block_contact_order'  => 1,
			'item_block_contact_position' => 'sidebar',
			'item_block_customfields_show' => 1,
			'item_block_customfields_order' => 2,
			'item_block_map_show'       => 1,
			'item_block_map_order'      => 3,
			'item_block_address_show'   => 1,
			'item_block_address_order'  => 4,
			'event_block_contact_show'  => 1,
			'event_block_contact_order' => 1,
			'event_block_contact_position' => 'sidebar',
			'event_block_customfields_show' => 1,
			'event_block_customfields_order' => 2,
			'event_block_map_show'      => 1,
			'event_block_map_order'     => 3,
			'event_block_address_show'  => 1,
			'event_block_address_order' => 4,
			'show_map'                  => 1,
			'map_position_items'        => 'top',
			'map_position_item'         => 'sidebar',
			'map_default_zoom'          => 13,
			'address_format'            => 'international',
			'address_show_street_number' => 1,
			'address_show_postal_code'  => 1,
			'address_show_country'      => 1,
			'events_enabled'            => 1,
			'events_root_category'      => 0,
			'recurring_events_enabled'  => 0,
			'events_date_format'        => 'DATE_FORMAT_LC4',
			'events_time_format'        => 'H:i',
			'sef_ids_in_url'            => 1,
			'sef_add_canonical'         => 1,
			'robots_fallback'           => '',
			'meta_desc_fallback'        => 1,
		];
	}

	/**
	 * Get or create instance ID (UUID)
	 *
	 * @return  string  Instance ID (UUID)
	 *
	 * @since   1.0.0
	 */
	private function getOrCreateInstanceId(): string
	{
		$db = Factory::getDbo();
		
		// Try to get existing instance_id from component params
		$query = $db->getQuery(true)
			->select($db->quoteName('params'))
			->from($db->quoteName('#__extensions'))
			->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
			->where($db->quoteName('type') . ' = ' . $db->quote('component'));
		$db->setQuery($query);
		$paramsJson = $db->loadResult();
		
		if ($paramsJson) {
			$params = json_decode($paramsJson, true);
			if (isset($params['instance_id']) && !empty($params['instance_id'])) {
				return $params['instance_id'];
			}
		}
		
		// Generate new UUID v4
		$data = random_bytes(16);
		$data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Set version to 0100
		$data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Set bits 6-7 to 10
		$instanceId = vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
		
		// Save instance_id to component params
		$params = $paramsJson ? json_decode($paramsJson, true) : [];
		$params['instance_id'] = $instanceId;
		
		$query = $db->getQuery(true)
			->update($db->quoteName('#__extensions'))
			->set($db->quoteName('params') . ' = ' . $db->quote(json_encode($params)))
			->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
			->where($db->quoteName('type') . ' = ' . $db->quote('component'));
		
		try {
			$db->setQuery($query)->execute();
		} catch (\Exception $e) {
			$this->log('ERROR saving instance_id: ' . $e->getMessage());
		}
		
		return $instanceId;
	}

	/**
	 * Get existing instance ID
	 *
	 * @return  string|null  Instance ID or null if not found
	 *
	 * @since   1.0.0
	 */
	private function getInstanceId(): ?string
	{
		$db = Factory::getDbo();
		
		$query = $db->getQuery(true)
			->select($db->quoteName('params'))
			->from($db->quoteName('#__extensions'))
			->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
			->where($db->quoteName('type') . ' = ' . $db->quote('component'));
		$db->setQuery($query);
		$paramsJson = $db->loadResult();
		
		if ($paramsJson) {
			$params = json_decode($paramsJson, true);
			if (isset($params['instance_id']) && !empty($params['instance_id'])) {
				return $params['instance_id'];
			}
		}
		
		return null;
	}

	/**
	 * Get component version
	 *
	 * @return  string  Component version
	 *
	 * @since   1.0.0
	 */
	private function getComponentVersion(): string
	{
		$db = Factory::getDbo();
		
		$query = $db->getQuery(true)
			->select($db->quoteName('manifest_cache'))
			->from($db->quoteName('#__extensions'))
			->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
			->where($db->quoteName('type') . ' = ' . $db->quote('component'));
		$db->setQuery($query);
		$manifestCache = $db->loadResult();
		
		if ($manifestCache) {
			$manifest = json_decode($manifestCache);
			if ($manifest && isset($manifest->version)) {
				return $manifest->version;
			}
		}
		
		return 'unknown';
	}

	/**
	 * Send installation statistics to server
	 *
	 * @param   string  $action      Action type: 'install' or 'uninstall'
	 * @param   string  $instanceId  Instance ID (UUID)
	 * @param   string  $version     Component version
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function sendInstallationStats(string $action, string $instanceId, string $version): void
	{
		$this->log('sendInstallationStats: action=' . $action . ', instanceId=' . substr($instanceId, 0, 8) . '..., version=' . $version);
		
		try {
			// Validate inputs
			if (empty($action) || !in_array($action, ['install', 'uninstall'])) {
				$this->log('ERROR: Invalid action for stats: ' . $action);
				return;
			}
			
			if (empty($instanceId)) {
				$this->log('ERROR: Empty instance_id for stats');
				return;
			}
			
			if (empty($version)) {
				$this->log('ERROR: Empty version for stats');
				return;
			}
			
			$statsUrl = 'https://comdev.eu/stats/onecore-installation.php';
			$date = date('Y-m-d H:i:s');
			
			// Prepare data
			$data = [
				'action' => $action,
				'instance_id' => $instanceId,
				'version' => $version,
				'date' => $date,
			];
			
			$postData = http_build_query($data);
			$this->log('Sending POST to: ' . $statsUrl);
			$this->log('POST Data: ' . $postData);
			
			// Use cURL directly
			if (function_exists('curl_init')) {
				$ch = curl_init($statsUrl);
				curl_setopt_array($ch, [
					CURLOPT_POST => true,
					CURLOPT_POSTFIELDS => $postData,
					CURLOPT_RETURNTRANSFER => true,
					CURLOPT_FOLLOWLOCATION => false,
					CURLOPT_TIMEOUT => 5,
					CURLOPT_CONNECTTIMEOUT => 5,
					CURLOPT_SSL_VERIFYPEER => true,
					CURLOPT_HTTPHEADER => [
						'Content-Type: application/x-www-form-urlencoded',
						'Content-Length: ' . strlen($postData),
						'User-Agent: OneCore-Joomla/1.0'
					]
				]);
				
				$responseBody = curl_exec($ch);
				$statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
				$curlError = curl_error($ch);
				curl_close($ch);
				
				if ($curlError) {
					$this->log('cURL ERROR: ' . $curlError);
				} elseif ($statusCode !== 200) {
					$this->log('HTTP ERROR: Status ' . $statusCode . ', Response: ' . substr($responseBody, 0, 500));
				} else {
					$this->log('SUCCESS: Stats sent, Response: ' . substr($responseBody, 0, 200));
				}
			} else {
				$this->log('ERROR: cURL not available');
			}
		} catch (\Exception $e) {
			$this->log('EXCEPTION sending stats: ' . $e->getMessage());
		}
	}

}
