<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

\defined('_JEXEC') or die;

use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;

// Get displayData (passed from view)
// Use getForm() method (standard Joomla way, same as joomla.edit.global.php)
$form = null;
if ($displayData && method_exists($displayData, 'getForm')) {
    $form = $displayData->getForm();
} elseif (isset($displayData->form)) {
    $form = $displayData->form;
}

$view = $displayData;

// Get values from form (only if form exists and is valid)
$lat = ($form && method_exists($form, 'getValue')) ? ($form->getValue('latitude') ?: '') : '';
$lng = ($form && method_exists($form, 'getValue')) ? ($form->getValue('longitude') ?: '') : '';
$address = ($form && method_exists($form, 'getValue')) ? ($form->getValue('address') ?: '') : '';

// Try to get data directly from the view/model
$latFromData = '';
$lngFromData = '';
if ($view && is_object($view) && property_exists($view, 'item') && is_object($view->item)) {
	$latFromData = $view->item->latitude ?? '';
	$lngFromData = $view->item->longitude ?? '';
}

// Use item data if form getValue is empty
if (empty($lat) && !empty($latFromData)) {
	$lat = $latFromData;
}
if (empty($lng) && !empty($lngFromData)) {
	$lng = $lngFromData;
}

// Get address format settings from component configuration
$params = \Joomla\CMS\Component\ComponentHelper::getParams('com_onecore');
$addressFormat = $params->get('address_format', 'international');
$showStreetNumber = (int) $params->get('address_show_street_number', 1) === 1;
$showPostalCode = (int) $params->get('address_show_postal_code', 1) === 1;
$showCountry = (int) $params->get('address_show_country', 1) === 1;

// Generate proxy URL for geocoding
$proxyUrl = Route::_('index.php?option=com_onecore&task=content.geocode&' . \Joomla\CMS\Session\Session::getFormToken() . '=1', false, Route::TLS_IGNORE, true);

// Marker icon URL
$markerUrl = \Joomla\CMS\Uri\Uri::root() . 'components/com_onecore/assets/images/marker.png';

// Load Leaflet CSS and JS
$document = \Joomla\CMS\Factory::getApplication()->getDocument();
$document->addStyleSheet('https://unpkg.com/leaflet@1.9.4/dist/leaflet.css');
$document->addScript('https://unpkg.com/leaflet@1.9.4/dist/leaflet.js');
?>
<!-- DEBUG: Location layout loaded -->
<div class="row">
	<div class="col-lg-12">
		<?php if ($form && is_object($form)) : ?>
			<?php echo $form->renderFieldset('location'); ?>
		<?php else : ?>
			<div class="alert alert-danger">ERROR: Form is not available. Type: <?php echo gettype($form); ?></div>
		<?php endif; ?>
		<div id="onecore-map-container" style="height: 400px; margin-top: 20px; border: 1px solid #ddd; border-radius: 4px;"></div>
		<button type="button" class="btn btn-primary mt-2" id="onecore-geocode-btn" onclick="if (typeof window.geocodeAddress === 'function') { window.geocodeAddress(); } else { alert('<?php echo Text::_('COM_ONECORE_GEOCODE_ADDRESS_LOADING'); ?>'); }"><?php echo Text::_('COM_ONECORE_GEOCODE_ADDRESS'); ?></button>
	</div>
</div>

<?php
// Add JavaScript for map and geocoding
$document->addScriptDeclaration("
(function() {
    'use strict';
    
    // Saved coordinates from PHP (fallback if form fields are empty)
    var savedLat = " . json_encode($lat) . ";
    var savedLng = " . json_encode($lng) . ";
    var markerIconUrl = " . json_encode($markerUrl) . ";
    
    var map, marker;
    var latField = document.getElementById('jform_latitude');
    var lngField = document.getElementById('jform_longitude');
    var addressField = document.getElementById('jform_address');
    var addressStreetField = document.getElementById('jform_address_street');
    var addressStreetNumberField = document.getElementById('jform_address_street_number');
    var addressPostalCodeField = document.getElementById('jform_address_postal_code');
    var addressCityField = document.getElementById('jform_address_city');
    var addressCountryField = document.getElementById('jform_address_country');
    
    // Address format settings
    var addressFormat = " . json_encode($addressFormat) . ";
    var showStreetNumber = " . ($showStreetNumber ? 'true' : 'false') . ";
    var showPostalCode = " . ($showPostalCode ? 'true' : 'false') . ";
    var showCountry = " . ($showCountry ? 'true' : 'false') . ";
    
    // Proxy URL for geocoding (avoids CORS issues)
    var geocodeProxyUrl = " . json_encode($proxyUrl) . ";
    
    // Function to format address according to selected format
    function formatAddress(addressData) {
        if (!addressData) {
            console.log('formatAddress: No address data provided');
            return '';
        }
        
        // Handle both reverse geocoding (object with address property) and forward geocoding (array)
        var addr = addressData.address || addressData;
        
        if (!addr) {
            console.log('formatAddress: No address property found', addressData);
            return '';
        }
        
        console.log('formatAddress: Processing address data', addr);
        
        // Extract address components with multiple fallbacks for different address formats
        var street = addr.road || addr.street || addr.street_name || addr.pedestrian || '';
        var streetNumber = addr.house_number || addr.house || addr.housenumber || '';
        var postalCode = addr.postcode || addr.postal_code || '';
        var city = addr.city || addr.town || addr.village || addr.municipality || addr.locality || addr.city_district || '';
        var country = addr.country || addr.country_code || '';
        
        console.log('formatAddress: Extracted components', {
            street: street,
            streetNumber: streetNumber,
            postalCode: postalCode,
            city: city,
            country: country
        });
        
        // Re-get fields in case they weren't available at initialization or DOM changed
        var streetField = document.getElementById('jform_address_street');
        var streetNumberField = document.getElementById('jform_address_street_number');
        var postalCodeField = document.getElementById('jform_address_postal_code');
        var cityField = document.getElementById('jform_address_city');
        var countryField = document.getElementById('jform_address_country');
        
        console.log('formatAddress: Field elements found', {
            streetField: !!streetField,
            streetNumberField: !!streetNumberField,
            postalCodeField: !!postalCodeField,
            cityField: !!cityField,
            countryField: !!countryField
        });
        
        // Fill individual fields - always fill all fields if available, regardless of display settings
        if (streetField) {
            streetField.value = street;
            console.log('formatAddress: Set street to:', street);
            // Trigger change event
            if (typeof jQuery !== 'undefined' && jQuery(streetField).length) {
                jQuery(streetField).val(street).trigger('change').trigger('input');
            } else {
                var changeEvent = new Event('change', { bubbles: true });
                streetField.dispatchEvent(changeEvent);
                var inputEvent = new Event('input', { bubbles: true });
                streetField.dispatchEvent(inputEvent);
            }
        } else {
            console.warn('formatAddress: Street field not found');
        }
        
        if (streetNumberField) {
            streetNumberField.value = streetNumber;
            console.log('formatAddress: Set street number to:', streetNumber);
            if (typeof jQuery !== 'undefined' && jQuery(streetNumberField).length) {
                jQuery(streetNumberField).val(streetNumber).trigger('change').trigger('input');
            } else {
                var changeEvent2 = new Event('change', { bubbles: true });
                streetNumberField.dispatchEvent(changeEvent2);
                var inputEvent2 = new Event('input', { bubbles: true });
                streetNumberField.dispatchEvent(inputEvent2);
            }
        } else {
            console.warn('formatAddress: Street number field not found');
        }
        
        if (postalCodeField) {
            postalCodeField.value = postalCode;
            console.log('formatAddress: Set postal code to:', postalCode);
            if (typeof jQuery !== 'undefined' && jQuery(postalCodeField).length) {
                jQuery(postalCodeField).val(postalCode).trigger('change').trigger('input');
            } else {
                var changeEvent3 = new Event('change', { bubbles: true });
                postalCodeField.dispatchEvent(changeEvent3);
                var inputEvent3 = new Event('input', { bubbles: true });
                postalCodeField.dispatchEvent(inputEvent3);
            }
        } else {
            console.warn('formatAddress: Postal code field not found');
        }
        
        if (cityField) {
            cityField.value = city;
            console.log('formatAddress: Set city to:', city);
            if (typeof jQuery !== 'undefined' && jQuery(cityField).length) {
                jQuery(cityField).val(city).trigger('change').trigger('input');
            } else {
                var changeEvent4 = new Event('change', { bubbles: true });
                cityField.dispatchEvent(changeEvent4);
                var inputEvent4 = new Event('input', { bubbles: true });
                cityField.dispatchEvent(inputEvent4);
            }
        } else {
            console.warn('formatAddress: City field not found');
        }
        
        if (countryField) {
            countryField.value = country;
            console.log('formatAddress: Set country to:', country);
            if (typeof jQuery !== 'undefined' && jQuery(countryField).length) {
                jQuery(countryField).val(country).trigger('change').trigger('input');
            } else {
                var changeEvent5 = new Event('change', { bubbles: true });
                countryField.dispatchEvent(changeEvent5);
                var inputEvent5 = new Event('input', { bubbles: true });
                countryField.dispatchEvent(inputEvent5);
            }
        } else {
            console.warn('formatAddress: Country field not found');
        }
        
        // Format main address field according to selected format
        var formatted = '';
        
        switch(addressFormat) {
            case 'international':
                // Format: Street Number, City, State Postal Code, Country
                if (street) {
                    formatted = (streetNumber && showStreetNumber) ? street + ' ' + streetNumber : street;
                }
                if (city) {
                    formatted += (formatted ? ', ' : '') + city;
                }
                if (postalCode && showPostalCode) {
                    formatted += (formatted ? ', ' : '') + postalCode;
                }
                if (country && showCountry) {
                    formatted += (formatted ? ', ' : '') + country;
                }
                break;
                
            case 'european':
                // Format: Street Number, Postal Code City, Country
                if (street) {
                    formatted = (streetNumber && showStreetNumber) ? street + ' ' + streetNumber : street;
                }
                if (postalCode && city && showPostalCode) {
                    formatted += (formatted ? ', ' : '') + postalCode + ' ' + city;
                } else if (city) {
                    formatted += (formatted ? ', ' : '') + city;
                }
                if (country && showCountry) {
                    formatted += (formatted ? ', ' : '') + country;
                }
                break;
                
            case 'simple':
            default:
                // Simple format: Street Number, City
                if (street) {
                    formatted = (streetNumber && showStreetNumber) ? street + ' ' + streetNumber : street;
                }
                if (city) {
                    formatted += (formatted ? ', ' : '') + city;
                }
                break;
        }
        
        return formatted;
    }
    
    // Helper function to make geocoding requests via proxy
    function geocodeRequest(params) {
        var url = geocodeProxyUrl;
        var queryParams = [];
        
        if (params.q) {
            queryParams.push('q=' + encodeURIComponent(params.q));
            queryParams.push('type=search');
        } else if (params.lat !== undefined && params.lon !== undefined) {
            queryParams.push('lat=' + encodeURIComponent(params.lat));
            queryParams.push('lon=' + encodeURIComponent(params.lon));
            queryParams.push('type=reverse');
        }
        
        if (queryParams.length > 0) {
            url += (url.indexOf('?') === -1 ? '?' : '&') + queryParams.join('&');
        }
        
        return fetch(url)
            .then(function(response) {
                if (!response.ok) {
                    return response.json().then(function(err) {
                        throw new Error(err.error || 'Network response was not ok: ' + response.status);
                    }).catch(function() {
                        throw new Error('Network response was not ok: ' + response.status);
                    });
                }
                return response.json();
            })
            .then(function(data) {
                // Check if response contains an error
                if (data && data.error) {
                    throw new Error(data.error);
                }
                return data;
            });
    }
    
    function initMap() {
        var container = document.getElementById('onecore-map-container');
        if (!container) return;
        
        // ALWAYS use form field values (current state), not savedLat/savedLng
        var latFieldValue = latField ? latField.value : '';
        var lngFieldValue = lngField ? lngField.value : '';
        
        var lat = null;
        var lng = null;
        
        // Use form field if not empty
        if (latFieldValue && latFieldValue !== '') {
            lat = parseFloat(latFieldValue);
        }
        if (lngFieldValue && lngFieldValue !== '') {
            lng = parseFloat(lngFieldValue);
        }
        
        // If fields are empty, use saved values from PHP and populate fields
        if (!lat && savedLat) {
            lat = parseFloat(savedLat);
            if (latField) {
                latField.value = savedLat;
            }
        }
        if (!lng && savedLng) {
            lng = parseFloat(savedLng);
            if (lngField) {
                lngField.value = savedLng;
            }
        }
        
        // Use coordinates if valid, otherwise use London as default
        var initialLat = (lat && !isNaN(lat)) ? lat : 51.5074;
        var initialLng = (lng && !isNaN(lng)) ? lng : -0.1278;
        
        map = L.map('onecore-map-container').setView([initialLat, initialLng], 13);
        
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '&copy; <a href=\"https://www.openstreetmap.org/copyright\">OpenStreetMap</a> contributors',
            maxZoom: 19
        }).addTo(map);
        
        // Custom marker icon (original proportions: 360x541)
        var customIcon = L.icon({
            iconUrl: markerIconUrl,
            iconSize: [32, 48],
            iconAnchor: [16, 48],
            popupAnchor: [0, -48]
        });
        
        // Always add marker - if coordinates exist, use them, otherwise use default location
        marker = L.marker([initialLat, initialLng], {
            draggable: true,
            icon: customIcon
        }).addTo(map);
        
        // If coordinates exist, show them in popup
        if (latField?.value && lngField?.value) {
            marker.bindPopup('Lat: ' + initialLat.toFixed(6) + '<br>Lng: ' + initialLng.toFixed(6)).openPopup();
        } else {
            marker.bindPopup('Kliknij na mapie lub przeciągnij pinezkę, aby ustawić lokalizację');
        }
        
        // Make marker draggable
        marker.on('dragend', function(e) {
            var position = marker.getLatLng();
            var lat = position.lat;
            var lng = position.lng;
            
            // Re-get fields in case they're not in scope
            var currentLatField = document.getElementById('jform_latitude');
            var currentLngField = document.getElementById('jform_longitude');
            
            if (currentLatField) {
                currentLatField.value = lat.toFixed(8);
            }
            
            if (currentLngField) {
                currentLngField.value = lng.toFixed(8);
            }
            
            // Reverse geocoding via proxy
            geocodeRequest({lat: lat, lon: lng})
                .then(function(data) {
                    if (data && data.address) {
                        var formatted = formatAddress(data);
                        if (addressField && formatted) {
                            addressField.value = formatted;
                            // Trigger change event
                            if (typeof jQuery !== 'undefined' && jQuery(addressField).length) {
                                jQuery(addressField).trigger('change').trigger('input');
                            } else {
                                var changeEvent = new Event('change', { bubbles: true });
                                addressField.dispatchEvent(changeEvent);
                            }
                        }
                    }
                })
                .catch(function(err) {
                    console.log('Geocoding error:', err);
                });
        });
        
        map.on('click', function(e) {
            var lat = e.latlng.lat;
            var lng = e.latlng.lng;
            
            // Re-get fields
            var currentLatField = document.getElementById('jform_latitude');
            var currentLngField = document.getElementById('jform_longitude');
            
            if (currentLatField) {
                currentLatField.value = lat.toFixed(8);
            }
            
            if (currentLngField) {
                currentLngField.value = lng.toFixed(8);
            }
            
            if (marker) {
                marker.setLatLng([lat, lng]);
                marker.bindPopup('Lat: ' + lat.toFixed(6) + '<br>Lng: ' + lng.toFixed(6)).openPopup();
            } else {
                marker = L.marker([lat, lng], {draggable: true, icon: customIcon}).addTo(map);
                marker.bindPopup('Lat: ' + lat.toFixed(6) + '<br>Lng: ' + lng.toFixed(6)).openPopup();
            }
            
            // Reverse geocoding via proxy
            geocodeRequest({lat: lat, lon: lng})
                .then(function(data) {
                    if (data && data.address) {
                        var formatted = formatAddress(data);
                        if (addressField && formatted) {
                            addressField.value = formatted;
                            // Trigger change event
                            if (typeof jQuery !== 'undefined' && jQuery(addressField).length) {
                                jQuery(addressField).trigger('change').trigger('input');
                            } else {
                                var changeEvent = new Event('change', { bubbles: true });
                                addressField.dispatchEvent(changeEvent);
                            }
                        }
                    }
                })
                .catch(function(err) {
                    console.log('Geocoding error:', err);
                });
        });
    }
    
    // Make geocodeAddress available globally
    window.geocodeAddress = function() {
        // Re-get fields in case they weren't available at initialization
        var currentAddressField = document.getElementById('jform_address');
        var currentLatField = document.getElementById('jform_latitude');
        var currentLngField = document.getElementById('jform_longitude');
        
        console.log('Geocoding - Address field:', currentAddressField);
        console.log('Geocoding - Latitude field:', currentLatField);
        console.log('Geocoding - Longitude field:', currentLngField);
        
        if (!currentAddressField) {
            console.error('Address field not found');
            alert('Pole adresu nie zostało znalezione');
            return;
        }
        
        if (!currentLatField) {
            console.error('Latitude field not found');
        }
        
        if (!currentLngField) {
            console.error('Longitude field not found');
        }
        
        var address = currentAddressField.value;
        
        if (!address || address.trim() === '') {
            alert('Proszę wpisać adres do geokodowania');
            currentAddressField.focus();
            return;
        }
        
        var query = address.trim();
        
        // Show loading indicator
        var btn = document.getElementById('onecore-geocode-btn') || document.querySelector('button[onclick*=\"geocodeAddress\"]');
        var originalText = btn ? btn.textContent : '';
        if (btn) {
            btn.disabled = true;
            btn.textContent = 'Geokodowanie...';
        }
        
        geocodeRequest({q: query})
            .then(function(response) { 
                return response; 
            })
            .then(function(data) {
                if (data && data.length > 0) {
                    var lat = parseFloat(data[0].lat);
                    var lng = parseFloat(data[0].lon);
                    
                    if (isNaN(lat) || isNaN(lng)) {
                        throw new Error('Invalid coordinates received');
                    }
                    
                    // Update latitude field
                    if (currentLatField) {
                        var latValue = lat.toFixed(8);
                        currentLatField.value = latValue;
                        console.log('Set latitude to:', latValue);
                        
                        // Use jQuery if available (Joomla often uses jQuery)
                        if (typeof jQuery !== 'undefined' && jQuery(currentLatField).length) {
                            jQuery(currentLatField).val(latValue).trigger('change').trigger('input');
                        } else {
                            // Native events
                            var changeEvent = new Event('change', { bubbles: true });
                            currentLatField.dispatchEvent(changeEvent);
                            var inputEvent = new Event('input', { bubbles: true });
                            currentLatField.dispatchEvent(inputEvent);
                        }
                        
                        // Force update
                        currentLatField.setAttribute('value', latValue);
                    } else {
                        console.error('Latitude field not found');
                        alert('Pole szerokości geograficznej nie zostało znalezione');
                    }
                    
                    // Update longitude field
                    if (currentLngField) {
                        var lngValue = lng.toFixed(8);
                        currentLngField.value = lngValue;
                        console.log('Set longitude to:', lngValue);
                        
                        // Use jQuery if available (Joomla often uses jQuery)
                        if (typeof jQuery !== 'undefined' && jQuery(currentLngField).length) {
                            jQuery(currentLngField).val(lngValue).trigger('change').trigger('input');
                        } else {
                            // Native events
                            var changeEvent2 = new Event('change', { bubbles: true });
                            currentLngField.dispatchEvent(changeEvent2);
                            var inputEvent2 = new Event('input', { bubbles: true });
                            currentLngField.dispatchEvent(inputEvent2);
                        }
                        
                        // Force update
                        currentLngField.setAttribute('value', lngValue);
                    } else {
                        console.error('Longitude field not found');
                        alert('Pole długości geograficznej nie zostało znalezione');
                    }
                    
                    if (map) {
                        map.setView([lat, lng], 15);
                        
                        if (marker) {
                            marker.setLatLng([lat, lng]);
                            marker.bindPopup('Lat: ' + lat.toFixed(6) + '<br>Lng: ' + lng.toFixed(6)).openPopup();
                        } else {
                            marker = L.marker([lat, lng], {draggable: true, icon: customIcon}).addTo(map);
                            marker.bindPopup('Lat: ' + lat.toFixed(6) + '<br>Lng: ' + lng.toFixed(6)).openPopup();
                        }
                    }
                    
                    // Perform reverse geocoding to get address details
                    return geocodeRequest({lat: lat, lon: lng});
                } else {
                    alert('Nie znaleziono adresu. Spróbuj wpisać bardziej szczegółowy adres.');
                    return Promise.reject(new Error('Address not found'));
                }
            })
            .then(function(addressData) {
                // Format and fill address fields
                console.log('geocodeAddress: Received address data from reverse geocoding', addressData);
                if (addressData && addressData.address) {
                    var formatted = formatAddress(addressData);
                    console.log('geocodeAddress: Formatted address:', formatted);
                    if (currentAddressField && formatted) {
                        currentAddressField.value = formatted;
                        // Trigger change event
                        if (typeof jQuery !== 'undefined' && jQuery(currentAddressField).length) {
                            jQuery(currentAddressField).val(formatted).trigger('change').trigger('input');
                        } else {
                            var changeEvent = new Event('change', { bubbles: true });
                            currentAddressField.dispatchEvent(changeEvent);
                            var inputEvent = new Event('input', { bubbles: true });
                            currentAddressField.dispatchEvent(inputEvent);
                        }
                    }
                } else {
                    console.warn('geocodeAddress: No address data or address property in response', addressData);
                }
            })
            .catch(function(err) {
                console.error('Geocoding error:', err);
                alert('Błąd podczas geokodowania adresu: ' + (err.message || 'Nieznany błąd'));
            })
            .finally(function() {
                if (btn) {
                    btn.disabled = false;
                    btn.textContent = originalText;
                }
            });
    };
    
    var mapInitialized = false;
    
    function ensureMapInitialized() {
        if (typeof L === 'undefined') {
            console.error('Leaflet library not loaded');
            return;
        }
        
        if (!mapInitialized) {
            mapInitialized = true;
            initMap();
            // Invalidate size after initialization to fix rendering
            setTimeout(function() {
                if (map) {
                    map.invalidateSize();
                }
            }, 300);
        } else {
            // Map already initialized, just invalidate size
            setTimeout(function() {
                if (map) {
                    map.invalidateSize();
                }
            }, 100);
        }
    }
    
    // Initialize map when location tab is shown
    document.addEventListener('DOMContentLoaded', function() {
        // Wait for joomla-tab custom element to be defined
        customElements.whenDefined('joomla-tab').then(function() {
            var tabContainer = document.getElementById('myTab');
            var locationTabButton = tabContainer ? tabContainer.querySelector('button[aria-controls=\"location\"]') : null;
            var locationTabPane = document.getElementById('location');
            
            if (locationTabPane) {
                // Check if location tab is already active
                if (locationTabPane.hasAttribute('active')) {
                    ensureMapInitialized();
                }
                
                // Listen for Joomla tab shown event
                if (locationTabButton) {
                    locationTabButton.addEventListener('joomla.tab.shown', function() {
                        ensureMapInitialized();
                    });
                }
                
                // Also use MutationObserver as fallback
                var observer = new MutationObserver(function(mutations) {
                    if (locationTabPane.hasAttribute('active')) {
                        ensureMapInitialized();
                    }
                });
                
                observer.observe(locationTabPane, {
                    attributes: true,
                    attributeFilter: ['active'],
                    childList: false,
                    subtree: false
                });
            } else {
                // Fallback: try to initialize after a delay
                setTimeout(function() {
                    ensureMapInitialized();
                }, 500);
            }
        }).catch(function() {
            // Fallback if custom element is not available
            setTimeout(function() {
                var locationTabPane = document.getElementById('location');
                if (locationTabPane) {
                    ensureMapInitialized();
                }
            }, 500);
        });
    });
    
    // Also check if DOM is already loaded
    if (document.readyState === 'complete' || document.readyState === 'interactive') {
        setTimeout(function() {
            var locationTabPane = document.getElementById('location');
            if (locationTabPane && locationTabPane.hasAttribute('active')) {
                ensureMapInitialized();
            }
        }, 300);
    }
})();
");
